//****************************************************************************//
//  CyCrc                                                     version: 1.0.3  //
//  copyright (c)2025 by Adrian Hunt                https://CyApplication.uk  //
//****************************************************************************//
//                                                                            //
//  Optimised flexable CRC engine.  Support for various standards: simple to  //
//  use and extened.  If you're not fussed about the details, you can simply  //
//  start just by creating a default CRC object and adding data to it.        //
//                                                                            //
//    CyTools::crc32 a_crc_obj;                                               //
//                                                                            //
//    a_crc_obj.add(some_data);                                               //
//                                                                            //
//    std::cout << a_crc_obj;                                                 //
//                                                                            //
//  CyCrc adds two new  classes to the CyTools namespace  (a namespace I use  //
//  for all my common untility stuff.)  CyTools::crc and CyTools::crc_table,  //
//  both template classes  that take a single argument,  the CRC parameters.  //
//  CRC parameters  do not have a defined type,  but a requirement to have a  //
//  publicly defined set of a type and static constants.  CyCrc defines four  //
//  common standards,  one each for 8, 16, 32 and 64-bit CRC.  CyCrcDefs.hpp  //
//  defines more CRC standards, including one's like 5-bit USB.               //
//                                                                            //
//    CyTools::crc<crc32_pkzip_params> crc_a; // specific CRC standard        //
//    CyTools::crc<crc32_params> crc_b;       // default 32-bit CRC           //
//    CyTools::crc32 crc_c;                   // default 32-bit CRC           //
//                                                                            //
//  CyTools::crc can  be run with  or without a table:  calculations  with a  //
//  table can be more than two times faster.  By default, if CyTools::crc is  //
//  not  given a  table when  created,  then a  Just In Time  table  will be  //
//  created, if needed. CyTools::crc has the following basic constructors.    //
//                                                                            //
//    crc();                        // JIT table                              //
//    crc(bool AJitTable);          // true - JIT table; false - tableless    //
//    crc(const table_t* ATable);   // Table pointer: nullptr - tableless     //
//    crc(const table_t& ATable) ;  // Table reference                        //
//                                                                            //
//  The table_t type  is defined as  a CyTools::crc_table  with the same CRC  //
//  paremeters.  CyCrc defines default tables for  8, 16, 32 and 64-bit CRC.  //
//  CyTools::crc_table  can be declared as a  [static] const, for a compile-  //
//  time table; declared as a local  variable for stack storage; or, created  //
//  with the new operator for heap storage.                                   //
//                                                                            //
//    static const CyTools::crc32_table COMPILE_TIME_TABLE;                   //
//    CyTools::crc32_table stack_table;                                       //
//    CyTools::crc32_table* heap_table_ptr = new CyTools::crc32_table();      //
//                                                                            //
//  These can then be used to create a CyTools::crc object.                   //
//                                                                            //
//    CyTools::crc32 crc_a(COMPILE_TIME_TABLE);                               //
//    CyTools::crc32 crc_b(stack_table);                                      //
//    CyTools::crc32 crc_c(heap_table_ptr);                                   //
//                                                                            //
//  After instigating a CyTools::crc object you need to add data.  There are  //
//  three methods:  add, add_bits and  add_stream.  Any one or  more  can be  //
//  called to add data.  add and add_bits are overload with several variants  //
//  to add almost any type of data.  The overloads for  add break the values  //
//  in to a pointer and size in bytes.                                        //
//                                                                            //
//		add(const void* AData, size_t ASize);                                   //
//		add(const uint8_t* AData, size_t ASize);                                //
//                                                                            //
//  A one-size-fits-all add method takes a single value of any type.          //
//                                                                            //
//    int data_a = 3435;                                                      //
//    double data_b = 1.563;                                                  //
//    a_struct_t data_c = { 3, 6.43f, 256 };                                  //
//                                                                            //
//    crc.add(data_a);                                                        //
//    crc.add(data_b);                                                        //
//    crc.add(data_c);                                                        //
//                                                                            //
//  This can also be achieved with the stream style, shift-left operator.     //
//                                                                            //
//    crc << data_a << data_b << data_c;                                      //
//                                                                            //
//  When adding  custom  types,  be careful  not to  add types  that include  //
//  dynamic content, such as pointers  or std::string.  To handle  pointers,  //
//  another one-size method takes a  pointer and the count of items (this is  //
//  NOT the size in bytes).                                                   //
//                                                                            //
//    int some_data[] = { 8, 26, -218 };                                      //
//                                                                            //
//    crc.add(some_data, 3);                                                  //
//                                                                            //
//  Here a  4-byte  int would cause  12-bytes  to be added to  the CRC.  For  //
//  string types,  there are methods for c-string style types (these rely on  //
//  character  traits),  And  there is  an  overload  for  std::basic_string  //
//  descendants.                                                              //
//                                                                            //
//    wchar_t* some_cstr = L"A C-style string";                               //
//    std::string some_stdstr = "A std::basic_string type";                   //
//                                                                            //
//    crc.add(some_cstr);                                                     //
//    crc.add(some_stdstr);                                                   //
//                                                                            //
//  A methods exists for iterators.  It is specified with  beginning and end  //
//  iterators.  Items are added  one at a time,  allowing for  containers of  //
//  more complex types.                                                       //
//                                                                            //
//    float* ptr_data = { 0.5f, 1.2f, -8394.751f };                           //
//    float* ptr_end = ptr_data + 3;                                          //
//    std::vector<int> int_data = { 56, -12, 4096 };                          //
//    std::vector<std::string> str_data = { "String one", "String two" };     //
//                                                                            //
//    crc.add(ptr_data, ptr_end);                                             //
//    crc.add(int_data.begin(), int_data.end());                              //
//    crc.add(str_data.begin(), str_data.end());                              //
//                                                                            //
//  The add_bits  methods allow the  adding of data  that does  not fit byte  //
//  boundries. The methods take a value or pointer to a value, and the count  //
//  of bits to add to the CRC.                                                //
//                                                                            //
//    int some_data = 0x05;                                                   //
//                                                                            //
//    crc.add_bits(some_data, 4);                                             //
//                                                                            //
//  The final way to  add data is  for it to  be read from  an input stream.  //
//  When calling add_stream,  a count  can be given.  The specified count of  //
//  characters is read from the stream before returning. If unspecified, the  //
//  whole stream will be read. In both states, the function will return when  //
//  the input stream reports eof or bad.                                      //
//                                                                            //
//    std::ifstream source(filename, std::ios::binary);                       //
//                                                                            //
//    crc.add_stream(source, 4);  // reads only 4 bytes and adds them         //
//    crc.add_stream(source);     // adds the rest of the stream              //
//                                                                            //
//  To retrieve the  final CRC you  can call the  get_value method.  The crc  //
//  object implicitly casts to it's  parameters value_t type, with the final  //
//  CRC value.  There is also a std::basic_ostream operator allowing the CRC  //
//  value to be easily output and with standard formatting.                   //
//                                                                            //
//    uint64_t get_crc = crc.get_value();                                     //
//    uint64_t cast_crc = crc;                                                //
//                                                                            //
//    std::cout << std::hex << crc << std::endl;                              //
//                                                                            //
//  A clear method  is provided to  reset a crc back to  it's initial value.  //
//  The  set_value  method  sets the  crc value:  the value is  unfinalised,  //
//  allowing a CRC calculation to be continued.                               //
//                                                                            //
//****************************************************************************//


#ifndef CYCRC_HPP
#define CYCRC_HPP

#include <cstddef>           // size_t
#include <cstdint>           // uint8_t, uint_fast8_t etc.
#include <cstring>           // std::memcpy
#include <algorithm>         // std::min
#include <string>            // std::basic_string, std::char_traits
#include <iostream>          // std::basic_istream, std::basic_ostream
#include <type_traits>       // std::is_integal_v, std::is_unsigned_v, std::is_same_v

// validates CRC parameters
#define ASSERT_CRC_PARAMS(params)  \
  static_assert(std::is_integral_v<typename params::value_t> &&  \
                std::is_unsigned_v<typename params::value_t>, "CRC parameters invalid: value_t must be an unsigned integer type");  \
  static_assert((sizeof(typename params::value_t) * 8) >= 8, "CRC parameters invalid: value_t width must be at least 8-bits");  \
  static_assert(std::is_same_v<decltype(params::SIZE), const size_t>, "CRC parameters invalid: SIZE constant of type size_t must exist");  \
  static_assert(params::SIZE != 0, "CRC parameters invalid: CRC SIZE can not be 0");  \
  static_assert(params::SIZE <= (sizeof(typename params::value_t) * 8), "CRC parameters invalid: CRC SIZE can not be greater than value_t width");  \
  static_assert(std::is_same_v<decltype(params::POLYNOMIAL), const typename params::value_t>, "CRC parameters invalid: POLYNOMIAL constant of type value_t must exist");  \
  static_assert(params::POLYNOMIAL != 0, "CRC parameters invalid: CRC POLYNOMIAL can not be 0");  \
  static_assert(std::is_same_v<decltype(params::INIT_VALUE), const typename params::value_t>, "CRC parameters invalid: INIT_VALUE constant of type value_t must exist");  \
  static_assert(std::is_same_v<decltype(params::FINAL_XOR), const typename params::value_t>, "CRC parameters invalid: FINAL_XOR constant of type value_t must exist");  \
  static_assert(std::is_same_v<decltype(params::REFLECT_IN), const bool>, "CRC parameters invalid: REFLECT_IN constant of type bool must exist");  \
  static_assert(std::is_same_v<decltype(params::REFLECT_OUT), const bool>, "CRC parameters invalid: REFLECT_IN constant of type bool must exist");

// checks CRC parameter compatiblity
#define ASSERT_COMPATIBLE_CRC_PARAMS(params_a, params_b)  \
  static_assert((params_a::SIZE == params_b::SIZE) &&  \
                (params_a::POLYNOMIAL == params_b::POLYNOMIAL) &&  \
                (params_a::INIT_VALUE == params_b::INIT_VALUE) &&  \
                (params_a::REFLECT_IN == params_b::REFLECT_IN), "Incompatible CRC parameters");

namespace CyTools {

  // private namespace - util functions
	namespace CyCrc {

    // create a mask of bits
    template <class TYPE>
    inline constexpr TYPE bit_mask(size_t ABits) noexcept {
      TYPE high_bit = TYPE(1) << (ABits - 1);

      return high_bit | (high_bit - TYPE(1));
    }

    // create a mask of bits from CRC parameters
		template <class CRC_PARAMS>
		inline constexpr typename CRC_PARAMS::value_t mask() {
			return bit_mask<typename CRC_PARAMS::value_t>(CRC_PARAMS::SIZE);
		}

    // mask a value according to CRC parameters
		template <class CRC_PARAMS>
		inline constexpr typename CRC_PARAMS::value_t mask(const typename CRC_PARAMS::value_t& AValue) {
			return (AValue & mask<CRC_PARAMS>());
		}

    // reflect the bit order of a value: 1101 -> 1011
    template <class TYPE>
    constexpr TYPE reflect(TYPE AValue, size_t ACount) {
      TYPE result = 0;

      for (size_t bit = 0; bit < ACount; ++bit) {
        result = (result << 1) | (AValue & 0x01);
        AValue >>= 1;
      }

      return result;
    }

	}

  // CRC lookup table
  //   holds 256 lookup values of the crc engine
 	template <class CRC_PARAMS>
	class crc_table {
	public:
		ASSERT_CRC_PARAMS(CRC_PARAMS)

		typedef crc_table<CRC_PARAMS> self_t;
		typedef CRC_PARAMS params;
		typedef typename params::value_t value_t;

    // create a CRC table
		constexpr crc_table() noexcept {
      uint8_t byte = 0;
			do {
				_data[byte] = calc_lookup(byte);
			} while (++byte);
		}
    // copy table data from an identical table
		crc_table(const self_t& AOther) noexcept {
			std::memcpy(_data, AOther._data, sizeof(_data));
		}
    // copy table data from a compatible table
    template <class OTHER_PARAMS>
    crc_table(const crc_table<OTHER_PARAMS>& AOther) noexcept {
      ASSERT_COMPATIBLE_CRC_PARAMS(params, OTHER_PARAMS)

      uint8_t byte = 0;
      do {
        _data[byte] = AOther.lookup(byte);
      }
      while (++byte);
    }

    // no assignment
		self_t& operator = (const self_t&) = delete;

    // lookup a byte value
    inline const value_t& lookup(uint8_t AByte) const { return _data[AByte]; }
	private:
		value_t _data[256];

    // calculate CRC lookup value
		static constexpr value_t calc_lookup(const uint8_t AByte) noexcept {
      value_t value = AByte;

			if (params::REFLECT_IN) {
				value_t polynomial = CyCrc::reflect(params::POLYNOMIAL, params::SIZE);

				for (size_t bit = 0; bit < 8; ++bit)
         	value = (value >> 1) ^ ((value & 0x01) * polynomial);
			}
			else if (params::SIZE >= 8) {
        size_t shift = params::SIZE - 8;
        size_t size_less = params::SIZE - 1;

				value <<= shift;
				for (size_t bit = 0; bit < 8; ++bit)
          value = (value << 1) ^ (((value >> size_less) & 0x01) * params::POLYNOMIAL);
			}
			else {
        size_t shift = 8 - params::SIZE;
        value_t polynomial = params::POLYNOMIAL << shift;

				for (size_t bit = 0; bit < 8; ++bit)
         	value = (value << 1) ^ (((value >> 7) & 0x01) * polynomial);
			}

			return value;
		}

	};

  // main CRC object
	template <class CRC_PARAMS>
	class crc {
	public:
		ASSERT_CRC_PARAMS(CRC_PARAMS)

		typedef crc<CRC_PARAMS> self_t;
		typedef CRC_PARAMS params;
		typedef typename params::value_t value_t;
		typedef crc_table<params> table_t;

    // basic constructors
		constexpr crc() noexcept
			: _value(init_value()),
				_table(nullptr), _table_jit(true) {}
		explicit constexpr crc(bool AJitTable) noexcept
			: _value(init_value()),
				_table(nullptr), _table_jit(AJitTable) {}
		explicit constexpr crc(const table_t* ATable) noexcept
			: _value(init_value()),
				_table(ATable), _table_jit(false) {}
		explicit constexpr crc(const table_t& ATable) noexcept
			: _value(init_value()),
				_table(&ATable), _table_jit(false) {}

    // construction from a pre-calculated CRC value
		explicit constexpr crc(const value_t& ACrc, bool AJitTable = true) noexcept
			: _value(unfinalize(ACrc)),
				_table(nullptr), _table_jit(AJitTable) {}
		constexpr crc(const value_t& ACrc, const table_t* ATable) noexcept
			: _value(unfinalize(ACrc)),
				_table(ATable), _table_jit(false) {}
		constexpr crc(const value_t& ACrc, const table_t& ATable) noexcept
			: _value(unfinalize(ACrc)),
				_table(&ATable), _table_jit(false) {}

    // construction from identical crc
		constexpr crc(const self_t& AOther) noexcept
    	: _value(AOther._value),
      	_table(nullptr), _table_jit(true) {}
		constexpr crc(const self_t& AOther, bool AJitTable) noexcept
    	: _value(AOther._value),
      	_table(nullptr), _table_jit(AJitTable) {}
		constexpr crc(const self_t& AOther, const table_t* ATable) noexcept
    	: _value(AOther._value),
      	_table(ATable), _table_jit(false) {}
		constexpr crc(const self_t& AOther, const table_t& ATable) noexcept
    	: _value(AOther._value),
      	_table(&ATable), _table_jit(false) {}

    // construction from compatible crc
    template <class OTHER_PARAMS>
    explicit constexpr crc(const crc<OTHER_PARAMS>& AOther) noexcept
    	: _value(AOther._get_unfinalised()),
      	_table(nullptr), _table_jit(true) {
      ASSERT_COMPATIBLE_CRC_PARAMS(params, OTHER_PARAMS)
		}
    template <class OTHER_PARAMS>
    constexpr crc(const crc<OTHER_PARAMS>& AOther, bool AJitTable) noexcept
    	: _value(AOther._get_unfinalised()),
      	_table(nullptr), _table_jit(AJitTable) {
      ASSERT_COMPATIBLE_CRC_PARAMS(params, OTHER_PARAMS)
		}
    template <class OTHER_PARAMS>
    constexpr crc(const crc<OTHER_PARAMS>& AOther, const table_t* ATable) noexcept
    	: _value(AOther._get_unfinalised()),
      	_table(ATable), _table_jit(false) {
      ASSERT_COMPATIBLE_CRC_PARAMS(params, OTHER_PARAMS)
		}
    template <class OTHER_PARAMS>
    constexpr crc(const crc<OTHER_PARAMS>& AOther, const table_t& ATable) noexcept
    	: _value(AOther._get_unfinalised()),
      	_table(&ATable), _table_jit(false) {
      ASSERT_COMPATIBLE_CRC_PARAMS(params, OTHER_PARAMS)
		}

    // construction from rvalue referrence (takes table)
		crc(self_t&& ATmp) noexcept
    	: _value(ATmp._value),
      	_table(ATmp._table), _table_jit(ATmp._table_jit) {
      ATmp._table = nullptr;
    }

    // destructor, free JIT table
		~crc() {
			if (_table && _table_jit)
				delete _table;
		}

    // assignment from pre-calculated and identical crc
		inline self_t& operator = (const value_t& ACrc) { return set_value(ACrc); }
		inline self_t& operator = (const self_t& AOther) { _value = AOther._value; return *this; }

    // assignment from rvalue referrence (may take table)
		self_t& operator = (self_t&& ATmp) {
	   	_value = ATmp._value;

 			if (_table == nullptr) {
        _table = ATmp._table;
        _table_jit = ATmp._table_jit;

        ATmp._table = nullptr;
      }

      return *this;
    }

    // assignment from compatible crc
    template <class OTHER_PARAMS>
    inline self_t& operator = (const crc<OTHER_PARAMS>& AOther) {
      ASSERT_COMPATIBLE_CRC_PARAMS(params, OTHER_PARAMS)

      _value = AOther._get_unfinalised();
      return *this;
    }

    // cast to value_t with finalised value
		inline operator value_t () const { return get_value(); }

    // compares with idenitical crc
		inline bool operator == (const self_t& AOther) const { return (_value == AOther._value); }
		inline bool operator != (const self_t& AOther) const { return (_value != AOther._value); }

    // compares with compatible crc
    template <class OTHER_PARAMS>
    inline bool operator == (const crc<OTHER_PARAMS>& AOther) const {
      ASSERT_COMPATIBLE_CRC_PARAMS(params, OTHER_PARAMS)

      return (_value == AOther._get_unfinalised());
    }

    template <class OTHER_PARAMS>
    inline bool operator != (const crc<OTHER_PARAMS>& AOther) const {
      ASSERT_COMPATIBLE_CRC_PARAMS(params, OTHER_PARAMS)

      return (_value != AOther._get_unfinalised());
    }

    // stream style, left-shift operator to add data
		template <class TYPE>
		inline self_t& operator << (const TYPE& AData) { return add(AData); }

    // reset the crc to it's initial value
		inline self_t& clear() { _value = init_value(); return *this; }

    // add bytes of data to the crc, primary add method
		self_t& add(const uint8_t* AData, size_t ASize) {
			if ((AData != nullptr) && (ASize > 0)) {
				prep_table();

        if (_table)
					calc_crc(AData, ASize, _table, _value);
        else
        	calc_crc(AData, ASize, _value);
			}

			return *this;
		}

    // add a number of bits to the crc, primary add_bits method
		self_t& add_bits(const uint8_t* AData, size_t ACount) {
			if ((AData != nullptr) && (ACount > 0)) {
				size_t bytes = ACount / 8;
				size_t bits = ACount % 8;

				if (bytes > 0) {
					prep_table();

          if (_table)
						calc_crc(AData, bytes, _table, _value);
          else
            calc_crc(AData, bytes, _value);
        }
				if (bits > 0)
					calc_crc_bits(*(AData + bytes), bits, _value);
			}

			return *this;
		}

    // add untyped data to the crc
		inline self_t& add(const void* AData, size_t ASize) {
			return add((const uint8_t*) AData, ASize);
		}

    // add untype bits to the crc
		inline self_t& add_bits(const void* AData, size_t ACount) {
			return add_bits((const uint8_t*) AData, ACount);
		}

    // add a binary safe type to the crc
		template <class TYPE>
		inline self_t& add(const TYPE& AData) {
			return add((const uint8_t*) &AData, sizeof(TYPE));
		}

    // add a fixed number of bits from a binary safe type
		template <class TYPE>
		inline self_t& add_bits(const TYPE& AData, size_t ACount) {
			return add_bits((const uint8_t*) &AData, ACount);
		}

    // add a pointer to a type, buffer or array
		template <class TYPE>
		inline self_t& add(const TYPE* AData, size_t ACount) {
			return add((const uint8_t*) AData, ACount * sizeof(TYPE));
		}

    // add a number of bits from pointer to a type, buffer or array
		template <class TYPE>
		inline self_t& add_bits(const TYPE* AData, size_t ACount) {
			return add_bits((const uint8_t*) AData, ACount);
		}

    // add a c-string to the crc (null terminator not included)
    template <class CHAR_TYPE, class TRAITS = std::char_traits<CHAR_TYPE>>
    inline self_t& add(const CHAR_TYPE* AStr) {
      if (AStr == nullptr)
        return *this;

      return add(AStr, TRAITS::length(AStr));
    }

    // c-string helpers for signed and unsigned char
    inline self_t& add(const signed char* AStr) { return add((const char*) AStr); }
    inline self_t& add(const unsigned char* AStr) { return add((const char*) AStr); }

    // add a std::basic_string type to the crc
    template <class CHAR_TYPE, class TRAITS, class ALLOCATOR>
    inline self_t& add(const std::basic_string<CHAR_TYPE, TRAITS, ALLOCATOR>& AStr) {
      return add(AStr.c_str(), AStr.length());
    }

    // simple iterator support, add each item seperately
    template <class ITER_TYPE>
    self_t& add(ITER_TYPE ABegin, ITER_TYPE AEnd) {
      for (; ABegin != AEnd; ++ABegin)
        add(*ABegin);

      return *this;
    }

    // read data from a istream adding it to the crc
		template <class CHAR_TYPE, class TRAITS>
		self_t& add_stream(std::basic_istream<CHAR_TYPE, TRAITS>* AStream, typename std::basic_istream<CHAR_TYPE, TRAITS>::off_type ACount = -1) {
      static const size_t BUF_LEN = 1024 / sizeof(CHAR_TYPE);

      if ((ACount > 0) || (ACount == -1)) {
        CHAR_TYPE buffer[BUF_LEN];
        size_t get_count;

        prep_table();
        AStream->clear();

        if (ACount == -1) {
          if (_table) {
            // Whole stream, using a table
            while (!(AStream->eof() || AStream->bad())) {
              AStream->read(buffer, BUF_LEN);
              get_count = AStream->gcount();

              if (get_count > 0)
                calc_crc((const uint8_t*) buffer, get_count * sizeof(CHAR_TYPE), _table, _value);
            }
          }
          else {
            // Whole stream, tableless
            while (!(AStream->eof() || AStream->bad())) {
              AStream->read(buffer, BUF_LEN);
              get_count = AStream->gcount();

              if (get_count > 0)
                calc_crc((const uint8_t*) buffer, get_count * sizeof(CHAR_TYPE), _value);
            }
          }
        }
        else if (_table) {
          // Fixed length read, using a table
          while ((ACount > 0) && !(AStream->eof() || AStream->bad())) {
            AStream->read(buffer, std::min<size_t>(ACount, BUF_LEN));
            get_count = AStream->gcount();

            if (get_count > 0) {
              calc_crc((const uint8_t*) buffer, get_count * sizeof(CHAR_TYPE), _table, _value);
              ACount -= get_count;
            }
          }
        }
        else {
          // Fixed length read, tableless
          while ((ACount > 0) && !(AStream->eof() || AStream->bad())) {
            AStream->read(buffer, std::min<size_t>(ACount, BUF_LEN));
            get_count = AStream->gcount();

            if (get_count > 0) {
              calc_crc((const uint8_t*) buffer, get_count * sizeof(CHAR_TYPE), _value);
              ACount -= get_count;
            }
          }
        }
      }

			return *this;
		}

		template <class CHAR_TYPE, class TRAITS>
		self_t& add_stream(std::basic_istream<CHAR_TYPE, TRAITS>& AStream, typename std::basic_istream<CHAR_TYPE, TRAITS>::off_type ACount = -1) {
      return add_stream(&AStream, ACount);
    }

    // get and set the crc value
		inline value_t get_value() const { return finalize(_value); }
    inline self_t& set_value(const value_t& ACrc) { _value = unfinalize(ACrc); return *this; }

    inline const value_t& _get_unfinalised() const { return _value; }  // undocumented, internal!!
	private:
		value_t _value;
		const table_t* _table;
		bool _table_jit;

    // create a JIT table, if needed
		inline void prep_table() {
			if ((_table == nullptr) && _table_jit)
				_table = new table_t();
		}

    // calculate a crc from the given data, tableless
 		static void calc_crc(const uint8_t* AData, size_t ASize, value_t& AValue) noexcept {
			if (params::REFLECT_IN) {
				static const value_t POLYNOMIAL = CyCrc::reflect(params::POLYNOMIAL, params::SIZE);

				for (const uint8_t* end = AData + ASize; AData != end; ++AData) {
					AValue ^= *AData;

					for (size_t bit = 0; bit < 8; ++bit)
          	AValue = (AValue >> 1) ^ ((AValue & 0x01) * POLYNOMIAL);
				}
			}
			else if (params::SIZE >= 8) {
				static const size_t SHIFT = (params::SIZE >= 8) ? (params::SIZE - 8) : 0;
        static const size_t SIZE_LESS = params::SIZE - 1;

				for (const uint8_t* end = AData + ASize; AData != end; ++AData) {
					AValue ^= value_t(*AData) << SHIFT;

					for (size_t bit = 0; bit < 8; ++bit)
            AValue = (AValue << 1) ^ (((AValue >> SIZE_LESS) & 0x01) * params::POLYNOMIAL);
				}
			}
			else {
				static const size_t SHIFT = (params::SIZE < 8) ? (8 - params::SIZE) : 0;
				static const value_t POLYNOMIAL = params::POLYNOMIAL << SHIFT;

				for (const uint8_t* end = AData + ASize; AData != end; ++AData) {
					AValue ^= *AData;

					for (size_t bit = 0; bit < 8; ++bit)
          	AValue = (AValue << 1) ^ (((AValue >> 7) & 0x01) * POLYNOMIAL);
				}
			}
		}

    // calculate a crc from the given data, with table
		static void calc_crc(const uint8_t* AData, size_t ASize, const table_t* ATable, value_t& AValue) noexcept {
 			if (params::SIZE <= 8) {
        for (const uint8_t* end = AData + ASize; AData != end; ++AData)
          AValue = ATable->lookup(AValue ^ *AData);
      }
			else if (params::REFLECT_IN) {
				for (const uint8_t* end = AData + ASize; AData != end; ++AData)
					AValue = (AValue >> 8) ^ ATable->lookup(AValue ^ *AData);
			}
			else {
				static const size_t SHIFT = (params::SIZE >= 8) ? (params::SIZE - 8) : 0;

				for (const uint8_t* end = AData + ASize; AData != end; ++AData)
					AValue = (AValue << 8) ^ ATable->lookup((AValue >> SHIFT) ^ *AData);
  		}
		}

    // calculate a crc from the given number of bit, always tableless
		static void calc_crc_bits(uint8_t AData, size_t ACount, value_t& AValue) noexcept {
      AData &= CyCrc::bit_mask<uint8_t>(ACount);

			if (params::REFLECT_IN) {
				static const value_t POLYNOMIAL = CyCrc::reflect(params::POLYNOMIAL, params::SIZE);

				AValue ^= AData;
				for (size_t bit = 0; bit < ACount; ++bit)
          AValue = (AValue >> 1) ^ ((AValue & 0x01) * POLYNOMIAL);
			}
			else if (params::SIZE >= 8) {
        static const size_t SIZE_LESS = params::SIZE - 1;
	 			static const size_t SHIFT = (params::SIZE >= 8) ? (params::SIZE - 8) : 0;

				AValue ^= value_t(AData) << SHIFT;
				for (size_t bit = 0; bit < ACount; ++bit)
          AValue = (AValue << 1) ^ (((AValue >> SIZE_LESS) & 0x01) * params::POLYNOMIAL);
			}
			else {
				static const size_t SHIFT = (params::SIZE < 8) ? (8 - params::SIZE) : 0;
				static const value_t POLYNOMIAL = params::POLYNOMIAL << SHIFT;

				AValue ^= AData << (8 - ACount);     // TODO : or undo -- AData shift
				for (size_t bit = 0; bit < ACount; ++bit)
          AValue = (AValue << 1) ^ (((AValue >> 7) & 0x01) * POLYNOMIAL);
			}
		}

    // calculate the crc initial value
    static constexpr value_t init_value() noexcept {
      value_t value = params::INIT_VALUE;

      if ((!params::REFLECT_IN) && (params::SIZE < 8))
				value <<= (params::SIZE < 8) ? (8 - params::SIZE) : 0;

      return value;
    }

    // calculate the crc final value
		static constexpr value_t finalize(value_t AValue) noexcept {
      if ((!params::REFLECT_IN) && (params::SIZE < 8))
				AValue >>= (params::SIZE < 8) ? (8 - params::SIZE) : 0;

			if (params::REFLECT_OUT != params::REFLECT_IN)
				AValue = CyCrc::reflect(AValue, params::SIZE);

			return CyCrc::mask<params>(AValue ^ params::FINAL_XOR);
		}

    // reverse the final crc calculation
		static constexpr value_t unfinalize(value_t AValue) noexcept {
			AValue = CyCrc::mask<params>(AValue) ^ params::FINAL_XOR;

			if (params::REFLECT_OUT != params::REFLECT_IN)
				AValue = CyCrc::reflect(AValue, params::SIZE);

      if ((!params::REFLECT_IN) && (params::SIZE < 8))
				AValue <<= (params::SIZE < 8) ? (8 - params::SIZE) : 0;

			return AValue;
		}

	};

	/* template params structure...
	struct crc_params {
		typedef uint_fast??_t value_t;

		static const size_t SIZE;        // Bit width
		static const value_t POLYNOMIAL; // CRC polynomial
		static const value_t INIT_VALUE; // Initial CRC value
		static const value_t FINAL_XOR;  // Final XOR value
		static const bool REFLECT_IN;    // Reflect all input bits
		static const bool REFLECT_OUT;   // Reflect the output CRC
	};	*/

	// common crc standards
	struct crc8_hdlc_params {
		typedef uint_fast8_t value_t;

		static const size_t SIZE        = 8;
		static const value_t POLYNOMIAL = 0x07;
		static const value_t INIT_VALUE = 0xFF;
		static const value_t FINAL_XOR  = 0xFF;
		static const bool REFLECT_IN    = true;
		static const bool REFLECT_OUT   = true;
	};

	struct crc16_arc_params {
		typedef uint_fast16_t value_t;

		static const size_t SIZE        = 16;
		static const value_t POLYNOMIAL = 0x8005;
		static const value_t INIT_VALUE = 0x0000;
		static const value_t FINAL_XOR  = 0x0000;
		static const bool REFLECT_IN    = true;
		static const bool REFLECT_OUT   = true;
	};

	struct crc32_pkzip_params {
		typedef uint_fast32_t value_t;

		static const size_t SIZE        = 32;
		static const value_t POLYNOMIAL = 0x04C11DB7;
		static const value_t INIT_VALUE = 0xFFFFFFFF;
		static const value_t FINAL_XOR  = 0xFFFFFFFF;
		static const bool REFLECT_IN    = true;
		static const bool REFLECT_OUT   = true;
	};

	struct crc64_ecma_params {
		typedef uint_fast64_t value_t;

		static const size_t SIZE        = 64;
		static const value_t POLYNOMIAL = 0x42F0E1EBA9EA3693;
		static const value_t INIT_VALUE = 0x0000000000000000;
		static const value_t FINAL_XOR  = 0x0000000000000000;
		static const bool REFLECT_IN    = true;
		static const bool REFLECT_OUT   = true ;
	};

  // basic crc defines (defaults)
  typedef crc8_hdlc_params crc8_params;
  typedef crc16_arc_params crc16_params;
  typedef crc32_pkzip_params crc32_params;
  typedef crc64_ecma_params crc64_params;

  typedef crc_table<crc8_params> crc8_table;
  typedef crc_table<crc16_params> crc16_table;
  typedef crc_table<crc32_params> crc32_table;
  typedef crc_table<crc64_params> crc64_table;

  typedef crc<crc8_params> crc8;
  typedef crc<crc16_params> crc16;
  typedef crc<crc32_params> crc32;
  typedef crc<crc64_params> crc64;

}

// stream operator
template <class CHAR_TYPE, class TRAITS, class PARAMS>
inline std::basic_ostream<CHAR_TYPE, TRAITS>& operator << (std::basic_ostream<CHAR_TYPE, TRAITS>& AStream, const CyTools::crc<PARAMS>& ACrc) {
	return (AStream << ACrc.get_value());
}

#endif

