//****************************************************************************//
//  CyCarousel                                                version: 5.1.0  //
//  copyright (c)2025 by Adrian Hunt                https://CyApplication.uk  //
//****************************************************************************//
//                                                                            //
//                                                                            //
//****************************************************************************//
//                                                                            //
//  v1.0.0 - test bake    - anonymous object; wired into HTML                 //
//  v2.0.0 - acceptable   - convertion to constructor function                //
//  v2.0.1 - review       - clean-up, added CONSTANTs & stateStr              //
//  v2.1.1 - new features - added start, stop & basic move.. methods          //
//  v2.1.2 - missed bits  - added moveFirst, moveLast & moveBy                //
//  v3.0.0 - overhaul     - conversion to a class, upgrade to multi-media     //
//  v4.0.0 - attainment   - rewrite from scratch! handles any HTML element    //
//  v5.0.0 - heavenly     - manager implementation                            //
//  v5.1.0 - disorder     - random advance and manager sync mode              //
//                                                                            //
//****************************************************************************//

class Carousel {
  // CONSTANT: possible states
  static STATE_ERROR   = 0;
  static STATE_RUNNING = 1;
  static STATE_STOPPED = 2;

  // CONSTANT: transition between pages
  static FX_NONE      = 0;
  static FX_FADE      = 1;
  static FX_XFADE     = 2;
  static FX_SCROLL_LR = 3;
  static FX_SCROLL_RL = 4;
  static FX_SCROLL_TB = 5;
  static FX_SCROLL_BT = 6;
  static FX_ZOOM_IN   = 7;
  static FX_ZOOM_OUT  = 8;
  static FX_CURTIANS  = 9;      // coming to later review

  static #FX__FIRST = Carousel.FX_NONE;
  static #FX__LAST  = Carousel.FX_ZOOM_OUT;

  // value used for page properties
  static USE_DEFAULT = { text: "use default" };

  // classes fields
  static #styles     = null;
  static #globalMngr = null;

  static get globalMngr() {
    let manager = Carousel.#globalMngr;
    if (manager === null)
      manager = Carousel.#globalMngr = new CarouselManager();

    return manager;
  }

  // object fields
  #localMngr    = null;
  #state        = Carousel.STATE_ERROR;
  #frame        = null;
  #poster       = { elem: null, delay: -1 };
  #curtains     = { left: null, right: null, pause: -1 };
  #pages        = null;
  #page         = -1;
  #transIn      = { fx: -1, time: -1 };
  #transOut     = { fx: -1, time: -1 };
  #transTimer   = null;
  #delay        = -1;
  #updateTime   = -1;
  #random       = false;

  // public property access
  get manager() {
    let manager = this.#localMngr;
    if(manager === null)
      manager = Carousel.globalMngr;

    return manager;
  }

  get state() {
    return this.#state;
  }

  get stateStr() {
    let str;

    switch (this.state) {
    case STATE_ERROR:     str = "error";    break;
    case STATE_LOADING:   str = "loading";  break;
    case STATE_RUNNING:   str = "running";  break;
    case STATE_STOPPED:   str = "stopped";  break;
    default:
      str = "** undefined **";
    }

    return str;
  }

  get posterDelay() {
    return this.#poster.delay;
  }

  set posterDelay(time) {
    if (Number.isInteger(time) && (time >= 0))
      this.#poster.delay;
    else
      console.error("!- invalid posterDelay value, must be an integer, zero or greater -!");
  }

  get curtPause() {
    return this.#curtains.pause;
  }

  set curtPause(time) {
    if (Number.isInteger(time) && (time >= 0))
      this.#curtains.pause = time;
    else
      console.error("!- invalid curtPause value, must be an integer, zero or greater -!");
  }

  get current() {
    return this.#page;
  }

  set current(page) {
    if (Number.isInteger(value))
      this.moveTo(page);
    else
      console.error("!- invalid current value, must be an integer -!");
  }

  get transIn() {
    return Carousel.#transInterface(this.#transIn, false);
  }

  set transIn(trans) {
    Carousel.#assignTrans(trans, this.#transIn, false);
  }

  get transOut() {
    return Carousel.#transInterface(this.#transOut, false);
  }

  set transOut(trans) {
    Carousel.#assignTrans(trans, this.#transOut, false);
  }

  get delay() {
    return this.#delay;
  }

  set delay(millis) {
    if (Number.isInteger(millis) && (millis >= 0))
      this.#delay = millis;
    else
      console.error("!- invalid delay value, must be an integer, zero or greater -!");
  }

  get random() {
    return this.#random;
  }

  set random(value) {
    if (value === true) {
      if (this.#random === false)
        this.#chooseRandom();
    }
    else if (value === false)
      this.random = value;
    else
      console.error("!- invalid random value, must be a boolean -!");
  }

  get frame() {
    return this.#frame;
  }

  get pages() {
    let me = this;

    return {
      [Symbol.iterator]: function*() {
        for(let idx = 0; idx != me.#pages.length; idx++)
          yield Carousel.#pageInterface(me.#pages[idx]);
      },

      get length() {
        return me.#pages.length;
      },

      page: function(index) {
        let result = undefined;

        if ((!Number.isInteger(index)) || (index < 0) || (index >= me.#pages.length))
          console.error("!- invalid page index value -!");
        else
          result = Carousel.#pageInterface(me.#pages[index]);

        return result;
      }
    };
  }

  #chooseRandom() {
    let pages = this.#pages.length;

    if (this.#page == -1)
      this.#random = Math.floor(pages * Math.random());
    else {
      let steps = 1 + Math.floor((pages - 1) * Math.random())

      this.#random = (this.#page + steps) % pages;
    }
  }

  static #assignTrans(src, dest, allowDefault) {
    if (!("fx" in src))
      console.error("!- transition effect missing -!");
    else if (!("time" in src))
      console.error("!- transition time missing -!");
    else if (Carousel.#validateFx(src.fx, allowDefault) && Carousel.#validateTime(src.time, allowDefault)) {
      dest.fx = src.fx;
      dest.time = src.time;
    }
  }

  static #validateFx(fx, allowDefault) {
    let valid = false;

    if ((allowDefault && (fx === Carousel.USE_DEFAULT)) ||
        (Number.isInteger(fx) && (fx >= Carousel.#FX__FIRST) && (fx <= Carousel.#FX__LAST)))
      valid = true;
    else {
      let errMsg = "!- invalid transition effect: must be a Carousel.FX_... ";
      if (allowDefault)
        errMsg += "or Carousel.USE_DEFAULT ";
      errMsg +="value -!";

      console.error(errMsg);
    }

    return valid;
  }

  static #validateTime(time, allowDefault) {
    let valid = false;

    if ((allowDefault && (time === Carousel.USE_DEFAULT)) ||
        (Number.isInteger(time) && (time >= 0)))
      valid = true;
    else {
      let errMsg = "!- invalid transition time: must be an integer, zero or greater";
      if (allowDefault)
        errMsg += ", or Carousel.USE_DEFAULT";
      errMsg +=" -!";

      console.error(errMsg);
    }

    return valid;
  }

  static #transInterface(transObj, allowDefault) {
    return {
      get fx() {
        return transObj.fx;
      },

      set fx(value) {
        if (Carousel.#validateFx(value, allowDefault))
          transObj.fx = value;
      },

      get time() {
        return transObj.time;
      },

      set time(value) {
        if (Carousel.#validateTime(value, allowDefault))
          transObj.time = value;
      },

      valueOf() {
        return Carousel.#cloneTrans(transObj);
      },
    };
  }

  static #pageInterface(pageObj) {
    return {
      get index() {
        return pageObj.index;
      },

      get dom() {
        return pageObj.elem;
      },

      get transIn() {
        return Carousel.#transInterface(pageObj.transIn, true);
      },

      set transIn(trans) {
        Carousel.#assignTrans(trans, pageObj.transIn, true);
      },

      get transOut() {
        return Carousel.#transInterface(pageObj.transOut, true);
      },

      set transOut(trans) {
        Carousel.#assignTrans(trans, pageObj.transOut, true);
      },

      get delay() {
        return pageObj.delay;
      },

      set delay(millis) {
        if ((millis === Carousel.USE_DEFAULT) || (Number.isInteger(millis) && (millis >= 0)))
          pageObj.delay = millis;
        else
          console.error("!- invalid delay value, must be an integer value, zero or greater -!");
      },
    };
  }

  get [Symbol.toStringTag]() {
    return "Carousel";
  }

  constructor(domName, options = null) {
    console.debug("called - Carousel.constructor(...)");
    let okay = false;
    let frame = null;
    let poster = null;
    let curtains = null;
    let pages = null;

    // validate parameters
    if ((typeof(domName) !== "string") && !(domName instanceof String))
      console.error("!- parameter domName must be a string -!");
    else if ((options !== null) && !(options instanceof Object))
      console.error("!- parameter options must be an object or null -!");
    else {
      // types are right, validate deeper
      domName = domName.trim();

      if (domName == "")
        console.error("!- parameter domName cannot be an empty string -!");
      else {
        // validate specified dom object
        frame = document.getElementById(domName);

        if (frame === null)
          console.error("!- DOM object specified by domName cannot be found -!");
        else {
          // dom object okay, query pages
          pages = [...frame.children];

          // remove non-visual elements, the poster, and curtains
          for (let idx = pages.length - 1; idx >= 0; idx--) {
            let domTag = pages[idx];
            let tagName = domTag.tagName.toLowerCase();

            if ((tagName == "script") || (tagName == "style"))
              pages.splice(idx, 1);
            else {
              let type = domTag.getAttribute("data-carousel-type");

              if (type !== null) {
                type = type.trim().toLowerCase();

                if ((type != "") && (type != "page")) {
                  switch (type) {
                  case "poster":
                    if (poster === null) {
                      pages.splice(idx, 1);
                      poster = domTag;
                    }
                    else
                      console.warm("!- invalid data-carousel-type: poster redeclared, ignored -!");

                    break;
                  case "curtain":
                    if (curtains === null) {
                      pages.splice(idx, 1);
                      curtains = { single: domTag };
                    }
                    else
                      console.warm("!- invalid data-carousel-type: curtain redeclared, ignored -!");

                    break;
                  case "curtain-left":
                    if (curtains === null) {
                      pages.splice(idx, 1);
                      curtains = { left: domTag, right: null };
                    }
                    else if (("left" in curtains) && (curtains.left === null)) {
                      pages.splice(idx, 1);
                      curtains.left = domTag;
                    }
                    else
                      console.warm("!- invalid data-carousel-type: curtain redeclared, ignored -!");

                    break;
                  case "curtain-right":
                    if (curtains === null) {
                      pages.splice(idx, 1);
                      curtains = { left: null, right: domTag };
                    }
                    else if (("right" in curtains) && (curtains.right === null)) {
                      pages.splice(idx, 1);
                      curtains.right = domTag;
                    }
                    else
                      console.warn("!- invalid data-carousel-type: curtain redeclared, ignored -!");

                    break;
                  default:
                    console.warn("!- invalid data-carousel-type value: ignored -!");
                  };
                }
              }
            }
          }

          if (pages.length == 0)
            console.error("!- specified DOM element does not contian enough page elements (0) -!");
          else {
            if (pages.length == 1)
              console.warn("!- specified DOM element does not contian enough page elements (1) -!");

            okay = true;
          }
        }
      }
    }

    if (okay) {
      // validate options
      // default options values
      let optTmp = {
        autostart:   true,
        manager:     null,
        posterDelay: 2000,
        transFx:     Carousel.FX_XFADE,
        transTime:   500,
        delay:       10000,
        curtPause:   100,
        random:      false,
      };

      if (options === null) {
        // no options supplied, use defaults
        options = optTmp;
      }
      else {
        okay = false;

        // check each value found in the options object
        optCheck: {
          for (let key in options) {
            let value = null;

            // match options key name to valid options
            switch (key) {
            case "autostart":
              // make sure value is a bool
              value = options.autostart;
              if ((value === true) || (value === false))
                optTmp.autostart = value;
              else {
                console.error("!- invalid value for autostart option, must be boolean -!");
                break optCheck;
              }

              break;
            case "manager":
              // make sure we have a valid manage
              value = options.manager;
              if ((value === null) || (value instanceof CarouselManager))
                optTmp.manager = value;
              else {
                console.error("!- invalid value for manager option, must be a CarouselManager object -!");
                break optCheck;
              }

              break;
            case "transFx":
              // make sure value is a valid transition
              value = options.transFx;
              if (Number.isInteger(value) && (value >= Carousel.#FX__FIRST) && (value <= Carousel.#FX__LAST))
                optTmp.transFx = value;
              else {
                console.error("!- invalid value for transFx option, must be a Carousel.FX_... value -!");
                break optCheck;
              }

              break;
            case "transTime":
              // make sure value is an integer, greater than 0
              value = options.transTime;
              if (Number.isInteger(value) || (value > 0))
                optTmp.transTime = value;
              else {
                console.error("!- invalid value for transTime option, must be an integer greater than zero -!");
                break optCheck;
              }

              break;
            case "delay":
              // make sure value is an integer, greater than 0
              value = options.delay;
              if (Number.isInteger(value) || (value > 0))
                optTmp.delay = value;
              else {
                console.error("!- invalid value for delay option, must be an integer greater than zero -!");
                break optCheck;
              }

              break;
            case "posterDelay":
              // make sure value is an integer, 0 or greater
              value = options.posterDelay;
              if (Number.isInteger(value) || (value >= 0))
                optTmp.posterDelay = value;
              else {
                console.error("!- invalid value for posterDelay option, must be an integer, zero or greater -!");
                break optCheck;
              }

              break;
            case "curtPause":
              // make sure value is an integer, 0 or greater
              value = options.curtPause;
              if (Number.isInteger(value) || (value >= 0))
                optTmp.curtPause = value;
              else {
                console.error("!- invalid value for curtPause option, must be an integer, zero or greater -!");
                break optCheck;
              }

              break;
            case "random":
              // make sure value is a bool
              value = options.random;
              if ((value === true) || (value === false))
                optTmp.random = value;
              else {
                console.error("!- invalid value for random option, must be boolean -!");
                break optCheck;
              }

              break;
            default:
              console.warn("!- unknown option, " + key + "; ignored -!");
            }
          }

          // options passed validation
          options = optTmp;
          okay = true;
        }
      }
    }

    if (okay) {
      if (curtains === null)
        curtains = { left: null, right: null };
      else if ("single" in curtains) {
        let domLeft = curtains.single;
        let domRight = domLeft.cloneNode(true);

        frame.appendChild(domRight);

        curtains = { left: domLeft, right: domRight };
      }
      else if (curtains.left === null) {
        let domLeft = curtains.right.cloneNode(true);

        domLeft.style.scale = "-100% 100%";
        frame.appendChild(domLeft);

        curtains.left = domLeft;
      }
      else if (curtains.right === null) {
        let domRight = curtains.left.cloneNode(true);

        domRight.style.scale = "-100% 100%";
        frame.appendChild(domRight);

        curtains.right = domRight;
      }

      // validation and pre-processing complete
      // shoul be all plain-sailing from now on
      console.log("    carousel frame id: " + frame.id);
      console.log("    uses global manager: " + (options.manager === null));
      console.log("    has poster: " + (poster !== null));
      console.log("    pages: " + pages.length);

      // setup carousel object
      this.#localMngr = options.manager;
      this.#frame = frame;
      this.#poster = { elem: poster, delay: options.posterDelay };
      this.#curtains = { left: curtains.left, right: curtains.right, pause: options.curtPause };
      this.#pages = [];
      this.#transIn = { fx: options.transFx, time: options.transTime };
      this.#transOut = Carousel.#cloneTrans(this.#transIn);
      this.#delay = options.delay;
      this.#updateTime = 0;
      this.#random = false;
      
      // create internal style sheet, if needed
      if (Carousel.#styles === null)
        Carousel.#createStyles();

      // setup dom objects, add event handlers, and fill pages array
      frame.classList.add("Carousel");
      if (poster !== null)
        poster.classList.add("CarouselPoster");
      if (curtains.left !== null) {
        curtains.left.classList.add("CarouselCurtain");
        curtains.right.classList.add("CarouselCurtain");
      }

      let pageIdx = 0;
      for (let page of pages) {
        let pageType = page.tagName.toLowerCase();
        let pageIn = { fx: Carousel.USE_DEFAULT, time: Carousel.USE_DEFAULT };
        let pageOut = Carousel.#cloneTrans(pageIn);
        let pageDelay = Carousel.USE_DEFAULT;
        let value;

        Carousel.#resetPage(page);

        // set page CSS class
        page.classList.add("CarouselPage");

        // set properties and add event handlers based on tag type
        switch (pageType) {
        case "audio":
        case "video":
          page._ready = false;
          page._hold = false;
          page.addEventListener("canplay", function() { page._ready = true; });
          page.addEventListener("play", function() { page._hold = true; });
          page.addEventListener("ended", function() { page._hold = false; });
          page.ready = function() { return page._ready; };
          page.hold = function() { return page._hold; };
          page.preload = "auto";
          break;
        case "iframe":
          page._ready = false;
          page._hold = false;
          page.addEventListener("load", function() { page._ready = true; });
          page.ready = function() { return page._ready; };
          page.hold = function() { return page._hold; };
          break;
        case "img":
          page.ready = function() { return page.complete; };
          page.hold = function() { return false; };
          break;
        default:
          page.ready = function() { return true; };
          page.hold = function() { return false; };
        }

        value = page.getAttribute("data-carousel-trans");
        if ((value !== null) && (value != "")) {
          value = Carousel.#textToTrans(value);
          if (value === null)
            console.warn("!- invalid value in data-carousel-trans attribute; ignored -!");
          else {
            pageIn = value;
            pageOut = Carousel.#cloneTrans(value);
          }
        }
        else {
          value = page.getAttribute("data-carousel-trans-in");
          if ((value !== null) && (value != "")) {
            value = Carousel.#textToTrans(value);
            if (value === null)
              console.warn("!- invalid value in data-carousel-trans-in attribute; ignored -!");
            else
              pageIn = value;
          }

          value = page.getAttribute("data-carousel-trans-out");
          if ((value !== null) && (value != "")) {
            value = Carousel.#textToTrans(value);
            if (value === null)
              console.warn("!- invalid value in data-carousel-trans-out attribute; ignored -!");
            else
              pageOut = value;
          }
        }

        // check for, and validate, custom page attributes
        value = page.getAttribute("data-carousel-delay");
        if ((value !== null) && (value != "")) {
          value = Carousel.#textToTime(value);
          if (value === null)
            console.warn("!- invalid value in data-carousel-delay attribute; ignored -!");
          else
            pageDelay = value;
        }

        // add to pages array
        this.#pages[pageIdx] = {
          index:    pageIdx,
          elem:     page,
          transIn:  pageIn,
          transOut: pageOut,
          delay:    pageDelay,
        };

        pageIdx++;
      }

      // object construction complete
      // show poster or first page
      let startElem = poster;
      if (startElem === null) {
        let idx = 0;

        if (options.random)
          idx = Math.floor(this.#pages.length * Math.random());

        startElem = this.#pages[idx].elem;
        this.#page = idx;
      }
      else
        this.#page = -1;

      // setup for random jumping
      if (options.random)
        this.#chooseRandom();

      this.#showPage(startElem, { fx: Carousel.FX_NONE, time: 0 });
      this.#state = Carousel.STATE_STOPPED;

      // register with manager
      let me = this;
      this.manager.register({
        carousel: me,
        active: function() { return (me.#state == Carousel.STATE_RUNNING); },
        ready: function(time) { return me.#autoReady(time); },
        advance: function() { me.#autoAdvance(); },
      });

      if (options.autostart)
        this.start();
    }
  }

  static #cloneTrans(trans) {
    return { fx: trans.fx, time: trans.time };
  }

  static #createStyles() {
    let dom = document.createElement("style");
    let sheet = null;

    document.head.appendChild(dom);

    sheet = dom.sheet;                       -
    sheet.insertRule(".Carousel { " +
                       "overflow: clip; " +
                     "}", sheet.cssRules.length);
    sheet.insertRule(".CarouselPoster { " +
                       "display: block; " +
                       "position: absolute; " +
                       "z-index: 1000; " +
                       "max-width: 100%; " +
                       "max-height: 100%; " +
                       "left: 50%; " +
                       "top: 50%; " +
                       "translate: -50% -50%; " +
                     "}", sheet.cssRules.length);
    sheet.insertRule(".CarouselPage { " +
                       "display: none; " +
                       "position: absolute; " +
                       "max-width: 100%; " +
                       "max-height: 100%; " +
                       "left: 50%; " +
                       "top: 50%; " +
                       "translate: -50% -50%; " +
                     "}", sheet.cssRules.length);
    sheet.insertRule(".CarouselCurtain { " +
                       "display: none; " +
                       "position: absolute; " +
                       "width: 50%; " +
                       "height: 100%; " +
                       "top: 0; " +
                       "z-index: 9999999; " +
                     "}", sheet.cssRules.length);

    Carousel.#styles = sheet;
  }

  static #textToTime(text, allowDefault = true) {
    return Carousel.#strToTime(text.trim().toLowerCase(), allowDefault);
  }

  static #textToTrans(text, allowDefault = true) {
    let result = null;
    let fxValue = null;
    let timeValue = null;
    let space;

    text = text.trim().toLowerCase();
    space = text.indexOf(" ");

    if (idx != -1) {
      fxValue = Carousel.#strToFx(text.substring(0, space), allowDefault);
      timeValue = Carousel.#strToTime(text.substring(space + 1).trim(), allowDefault);
    }
    else {
      let value = Carousel.#strToFx(text, allowDefault);

      if (value !== null) {
        fxValue = value;
        timeValue = Carousel.USE_DEFAULT;
      }
      else {
        value = Carousel.#strToTime(text, allowDefault);

        if (value !== null) {
          fxValue = Carousel.USE_DEFAULT;
          timeValue = value;
        }
      }
    }

    if ((fxValue !== null) && (timeValue !== null))
      result = { fx: fxValue, time: timeValue };

    return  result;
  }

  static #strToFx(text, allowDefault = true) {
    let result = null;

    switch (text) {
    case "default":
      if (allowDefault)
        result = Carousel.USE_DEFAULT;

      break;
    case "none":       result = Carousel.FX_NONE;       break;
    case "fade":       result = Carousel.FX_FADE;       break;
    case "xfade":      result = Carousel.FX_XFADE;      break;
    case "scroll-lr":  result = Carousel.FX_SCROLL_LR;  break;
    case "scroll-rl":  result = Carousel.FX_SCROLL_RL;  break;
    case "scroll-tb":  result = Carousel.FX_SCROLL_TB;  break;
    case "scroll-bt":  result = Carousel.FX_SCROLL_BT;  break;
    case "zoom-in":    result = Carousel.FX_ZOOM_IN;    break;
    case "zoom-out":   result = Carousel.FX_ZOOM_OUT;   break;
    case "curtains":   result = Carousel.FX_CURTAINS;   break;
    }

    return result;
  }

  static #strToTime(text, allowDefault = true) {
    let result = null;

    if (allowDefault && (text == "default"))
      result = Carousel.USE_DEFAULT;
    else {
      // use a RegEx to validate and parse
      const regex = /^(\d{1,10}(?:\.\d{0,10})?)(ms|s|m|h)?$/;
      let parts = regex.exec(text);

      if (parts !== null) {
        let tmp = parseFloat(parts[1]);

        switch (parts[2]) {
        case "h":  tmp *= 3600000;  break;  // convert hours to millis
        case "m":  tmp *= 60000;    break;  // convert minutes to millis
        case "s":  tmp *= 1000;     break;  // convert seconds to millis
        }

        tmp = Math.round(tmp);
        if (tmp >= 1)
          result = tmp;
      }
    }

    return result;
  }

  #getTransIn(index) {
    let result = Carousel.#cloneTrans(this.#transIn);

    if (index != -1) {
      let tmp = this.#pages[index].transIn;

      if (tmp.fx !== Carousel.USE_DEFAULT)
        result.fx = tmp.fx;
      if (tmp.time !== Carousel.USE_DEFAULT)
        result.time = tmp.time;
    }

    return result;
  }

  #getTransOut(index) {
    let result = Carousel.#cloneTrans(this.#transOut);

    if (index != -1) {
      let tmp = this.#pages[index].transOut;

      if (tmp.fx !== Carousel.USE_DEFAULT)
        result.fx = tmp.fx;
      if (tmp.time !== Carousel.USE_DEFAULT)
        result.time = tmp.time;
    }

    return result;
  }

  #getDelay(index) {
    let result = this.#delay;

    if (index == -1)
      result = this.#poster.delay;
    else {
      let tmp = this.#pages[index].delay;

      if (tmp !== Carousel.USE_DEFAULT)
        result = tmp;
    }

    return result;
  }

  static #resetPage(elem) {
    elem.style.transition = "";
    elem.style.display = "none";
    elem.style.opacity = "100%";
    elem.style.left = "50%";
    elem.style.top = "50%";
    elem.style.scale = "100%";
    elem.style.zIndex = "0";
  }

  static #setPage(elem) {
    elem.style.transition = "";
    elem.style.display = "block";
    elem.style.opacity = "100%";
    elem.style.left = "50%";
    elem.style.top = "50%";
    elem.style.scale = "100%";
    elem.style.zIndex = "0";
  }

  #showPage(elem, trans) {
    switch (trans.fx) {
    case Carousel.FX_FADE:
    case Carousel.FX_XFADE:
      elem.style.opacity = "0%";
      elem.style.transition = "opacity " + trans.time + "ms ease-in-out";

      break;
    case Carousel.FX_SCROLL_LR:
      elem.style.left = "-150%";
      elem.style.transition = "left " + trans.time + "ms ease-in-out";

      break;
    case Carousel.FX_SCROLL_RL:
      elem.style.left = "250%";
      elem.style.transition = "left " + trans.time + "ms ease-in-out";

      break;
    case Carousel.FX_SCROLL_TB:
      elem.style.top = "-150%";
      elem.style.transition = "top " + trans.time + "ms ease-in-out";

      break;
    case Carousel.FX_SCROLL_BT:
      elem.style.top = "250%";
      elem.style.transition = "top " + trans.time + "ms ease-in-out";

      break;
    case Carousel.FX_ZOOM_IN:
      elem.style.scale = "0%";
      elem.style.zIndex = "100";
      elem.style.transition = "scale " + trans.time + "ms ease-in-out";

      break;
    case Carousel.FX_ZOOM_OUT:
      elem.style.scale = "10000%";
      elem.style.opacity = "0%";
      elem.style.zIndex = "100";
      elem.style.transition = "scale " + trans.time + "ms ease-in-out," +
                              "opacity " + trans.time + "ms ease-in-out";

      break;
    }

    elem.style.display = "block";

    let me = this;
    setTimeout(function() { me.#showPage00(elem, trans); }, 20);
  }

  #showPage00(elem, trans) {
    elem.style.opacity = "100%";
    elem.style.left = "50%";
    elem.style.top = "50%";
    elem.style.scale = "100%";

    if ((trans.fx === Carousel.FX_NONE) || (trans.time == 0))
      Carousel.#setPage(elem);
    else
      setTimeout(function() { Carousel.#setPage(elem); }, trans.time);
  }

  #hidePage(elem, trans) {
    let wait = 0;

    switch (trans.fx) {
    case Carousel.FX_NONE:
      elem.style.transition = "";

      break;
    case Carousel.FX_FADE:
      wait = trans.time;
    case Carousel.FX_XFADE:
      elem.style.transition = "opacity " + trans.time + "ms ease-in-out";
      elem.style.opacity = "0%";

      break;
    case Carousel.FX_SCROLL_LR:
      elem.style.transition = "left " + trans.time + "ms ease-in-out";
      elem.style.left = "250%";

      break;
    case Carousel.FX_SCROLL_RL:
      elem.style.transition = "left " + trans.time + "ms ease-in-out";
      elem.style.left = "-150%";

      break;
    case Carousel.FX_SCROLL_TB:
      elem.style.transition = "top " + trans.time + "ms ease-in-out";
      elem.style.top = "250%";

      break;
    case Carousel.FX_SCROLL_BT:
      elem.style.transition = "top " + trans.time + "ms ease-in-out";
      elem.style.top = "-150%";

      break;
    case Carousel.FX_ZOOM_IN:
      elem.style.transition = "scale " + trans.time + "ms ease-in-out," +
                              "opacity " + trans.time + "ms ease-in-out";
      elem.style.scale = "10000%";
      elem.style.opacity = "0%";
      elem.style.zIndex = "-100";

      break;
    case Carousel.FX_ZOOM_OUT:
      elem.style.transition = "scale " + trans.time + "ms ease-in-out";
      elem.style.scale = "0%";
      elem.style.zIndex = "-100";

      break;
    }

    if (trans.fx === Carousel.FX_NONE)
      Carousel.#resetPage(elem);
    else
      setTimeout(function() { Carousel.#resetPage(elem); }, trans.time);

    return wait;
  }

  #changePage(target, forward = true) {
    let curPage = this.#page;

    if (target != curPage) {
      let elem = null;
      let trans = null;
      let delayShow = 0;

      if (curPage == -1)
        elem = this.#poster.elem;
      else
        elem = this.#pages[curPage].elem;

      trans = this.#getTransOut(curPage);
      if (!forward)
        trans.fx = Carousel.#invertFx(trans.fx);

      delayShow = this.#hidePage(elem, trans);

      this.#page = target;
      this.#updateTime = Date.now() + this.#getDelay(target);

      if (this.#random !== false)
        this.#chooseRandom();

      if (this.#transTimer !== null) {
        clearTimeout(this.#transTimer);
        this.#transTimer = null;
      }

      elem = this.#pages[target].elem;
      trans = this.#getTransIn(target);
      if (!forward)
        trans.fx = Carousel.#invertFx(trans.fx);

      if (delayShow == 0)
        this.#showPage(elem, trans);
      else {
        let me = this;
        this.#transTimer = setTimeout(function() {
          me.#showPage(elem, trans);
          me.#transTimer = null;
        }, delayShow);
      }
    }
  }

  static #invertFx(fx) {
    switch (fx) {
    case Carousel.FX_SCROLL_LR:  fx = Carousel.FX_SCROLL_RL;  break;
    case Carousel.FX_SCROLL_RL:  fx = Carousel.FX_SCROLL_LR;  break;
    case Carousel.FX_SCROLL_TB:  fx = Carousel.FX_SCROLL_BT;  break;
    case Carousel.FX_SCROLL_BT:  fx = Carousel.FX_SCROLL_TB;  break;
    case Carousel.FX_ZOOM_IN:    fx = Carousel.FX_ZOOM_OUT;   break;
    case Carousel.FX_ZOOM_OUT:   fx = Carousel.FX_ZOOM_IN;    break;
    }

    return fx;
  }

  #autoReady(time) {
    let ready = false
    let currPage = null;

    if (this.#page != -1)
      currPage = this.#pages[this.#page];

    if ((currPage === null) || currPage.elem.ready()) {
      if (time >= this.#updateTime) {
        let nextPage;

        if (this.#random !== false)
          nextPage = this.#pages[this.#random];
        else if (currPage === null)
          nextPage = this.#pages[0];
        else {
          let idx = currPage.index + 1
          if (idx >= this.#pages.length)
            idx = 0;

          nextPage = this.#pages[idx];
        }

        ready = nextPage.elem.ready();
      }
    }

    return ready;
  }

  #autoAdvance() {
    if ((this.#page == -1) || !this.#pages[this.#page].elem.hold()) {
      if (this.#random === false)
        this.moveForward();
      else
        this.#changePage(this.#random, (this.#random > this.#page));
    }
  }

  start() {
    if ((this.#state == Carousel.STATE_STOPPED) && (this.#pages.length > 1)) {
      this.#updateTime = Date.now() + this.#getDelay(this.#page);
      this.#state = Carousel.STATE_RUNNING;
      this.manager.start();
    }
  }

  stop() {
    if (this.#state == Carousel.STATE_RUNNING)
      this.#state = Carousel.STATE_STOPPED;
  }

  moveTo(pageIdx, forward = undefined) {
    if (!Number.isInteger(pageIdx))
      console.error("!- invalid pageIdx parameter given, must be an integer -!");
    else if ((forward !== true) && (forward !== false) && (forward !== undefined))
      console.error("!- invalid forward parameter given, must be true or false, or left undefined -!");
    else {
      if (pageIdx < 0)
        pageIdx = 0;
      else if (pageIdx >= this.#pages.length)
        pageIdx = this.#pages.length - 1;

      if (forward === undefined)
        forward = (pageIdx > this.#page);

      this.#changePage(pageIdx, forward);
    }
  }

  moveBy(count) {
    if (!Number.isInteger(count))
      console.error("!- invalid count parameter given, must be an integer -!");
    else if (count != 0) {
      let pages = this.#pages.length;
      let forward = (count > 0);

      let index = (this.#page + count) % pages;
      if (index < 0)
        index = pages + index;

      this.#changePage(index, forward);
    }
  }

  moveRandom(forward = undefined) {
    if ((forward !== true) && (forward !== false) && (forward !== undefined))
      console.error("!- invalid forward parameter given, must be true or false, or left undefined -!");
    else {
      let pages = this.#pages.length;
      let index;

      if (this.#page == -1)
        index = Math.floor(pages * Math.random());
      else {
        let count = 1 + Math.floor((pages - 1) * Math.random());
        index = (this.#page + count) % pages;
      }

      if (forward === undefined)
        forward = (index > this.#page);

      this.#changePage(index, forward);
    }
  }

  moveForward() {
    let index = this.#page + 1;
    if (index >= this.#pages.length)
        index = 0;

    this.#changePage(index, true);
  }

  moveBackward() {
    let index = this.#page - 1;
    if (index < 0)
        index = this.#pages.length - 1;

    this.#changePage(index, false);
  }

  moveFirst() {
    this.#changePage(0, false);
  }

  moveLast() {
    this.#changePage(this.#pages.length - 1, true);
  }

}

class CarouselManager {
  // CONSTANT: possible states
  static STATE_ERROR   = 0;
  static STATE_RUNNING = 1;
  static STATE_STOPPED = 2;

  // CONSTANT: sync options
  static SYNC_NONE = 0;
  static SYNC_ALL  = 1;
  static SYNC_ONE  = 2;
  
  static #PING = 100;

  #state     = CarouselManager.STATE_ERROR;
  #carousels = null;
  #sync      = null;
  #timer     = null;

  get state() {
    return this.#state;
  }

  get stateStr() {
    let str;

    switch (this.#state) {
    case CarouselManager.STATE_ERROR:     str = "error";    break;
    case CarouselManager.STATE_RUNNING:   str = "running";  break;
    case CarouselManager.STATE_STOPPED:   str = "stopped";  break;
    default:
      str = "** undefined **";
    }

    return str;
  }

  get carousels() {
    let me = this;

    return {
      [Symbol.iterator]: function*() {
        for(const [ident, ifaceObj] of me.#carousels)
          yield ifaceObj.carousel;
      },

      get length() {
        return me.#carousels.size;
      },

      carousel: function(index) {
        let result = undefined;

        if ((!Number.isInteger(index)) || (index < 0) || (index >= me.#carousels.size))
          console.error("!- invalid carousel index value -!");
        else {
          const ifaceObjs = me.#carousels.values();
          const ourIface = ifaceObjs[index];

          result = ourIface.carousel;
        }

        return result;
      }
    };
  }

  get sync() {
    return this.#sync;
  }

  set sync(value) {
    if (Number.isInteger(value) && (value >= CarouselManager.SYNC_NONE) && (value <= CarouselManager.SYNC_ONE))
      this.#sync = value;
    else
      console.error("!- Invalid sync value: must be a CarouselManager.SYNC_... value -!");
  }

  get [Symbol.toStringTag]() {
    return "CarouselManager";
  }

  constructor() {
    this.#carousels = new Map();
    this.#sync = CarouselManager.SYNC_NONE;
    this.#timer = null;
    this.#state = CarouselManager.STATE_STOPPED;
  }

  #update() {
    let now = Date.now();

    if (this.#sync == CarouselManager.SYNC_NONE) {
      let active = false;

      for (const [ident, ifaceObj] of this.#carousels) {
        if (ifaceObj.active()) {
          active = true;

          if (ifaceObj.ready(now))
            ifaceObj.advance();
        }
      }

      if (!active)
        this.stop();
    }
    else {
      let active = [];
      let allReady = true;

      for (const [ident, ifaceObj] of this.#carousels) {
        if (ifaceObj.active()) {
          active.push(ifaceObj);

          if (!ifaceObj.ready(now)) {
            allReady = false;
            break;
          };
        }
      }

      if (active.length == 0)
        this.stop();
      else if (allReady) {
        if (this.#sync == CarouselManager.SYNC_ONE) {
          let sizes = [];
          let total = 0;
          let selection;

          for (let index = 0; index != active.length; index++)
            total += sizes[index] = active[index].carousel.pages.length;

          selection = Math.floor(total * Math.random());
          for (let index = 0; index != active.length; index++)
            if (selection >= sizes[index])
              selection -= sizes[index];
            else {
              active[index].advance();
              break;
            }
        }
        else
          for (const ifaceObj of active)
            ifaceObj.advance();
      }
    }
  }

  register(ifaceObj) {
    if (("carousel" in ifaceObj) && (ifaceObj.carousel instanceof Carousel) &&
        ("active" in ifaceObj) && (typeof(ifaceObj.active) == "function") &&
        ("ready" in ifaceObj) && (typeof(ifaceObj.ready) == "function") &&
        ("advance" in ifaceObj) && (typeof(ifaceObj.advance) == "function"))
      this.#carousels.set(ifaceObj.carousel.frame.id, ifaceObj);
    else
      console.error("!- interface object appears invalid -!");
  }

  unregister(ident) {
    if (ident instanceof Carousel)
      ident = ident.frame.id;

    this.#carousels.delete(ident);
  }

  start() {
    if (this.#timer === null) {
      let me = this;
      this.#timer = setInterval(function() {
        me.#update();
      }, CarouselManager.#PING);

      this.#state = CarouselManager.STATE_RUNNING;
    }
  }

  stop() {
    if (this.#timer !== null) {
      clearInterval(this.#timer);
      this.#timer = null;
      this.#state = CarouselManager.STATE_STOPPED;
    }
  }

}

