//****************************************************************************//
//  CyVersion                                                version: 1.0.0a  //
//  copyright (c)2024 by Adrian Hunt                https://CyApplication.uk  //
//****************************************************************************//

#ifndef CYVERSION_H
#define CYVERSION_H

#include <cstdlib>
#include <filesystem>
#include <winver.h>

namespace CyTools {

	class CyVersion {
	public:
		struct Language {
			WORD lang_id;
			WORD charset;

			std::string as_string() const;
		};

		CyVersion()
			: _ready(false), _buffer(nullptr), _buf_size(0),
				_lang_list(nullptr), _lang_count(0), _lang_sel(0), _info(nullptr) {}
		CyVersion(const std::filesystem::path& ABinName)
			: _ready(false), _buffer(nullptr), _buf_size(0),
				_lang_list(nullptr), _lang_count(0), _lang_sel(0), _info(nullptr) {
			load(ABinName);
		}
		~CyVersion() {
			if (_buffer != nullptr)
				std::free(_buffer);
		}

		inline bool ready() const { return _ready; }
		inline const std::filesystem::path& filename() const { return _filename; }

		bool load(const std::filesystem::path& ABinName);

		inline DWORD get_languages() const { return _lang_count; }
		inline const Language& get_lanuage(DWORD AIndex) const { return _lang_list[AIndex]; }
		inline void select_lang(DWORD AIndex) { _lang_sel = AIndex; }

		std::string fetch(const std::string& AName) const;

		inline std::string comments() const { return fetch("Comments"); }
		inline std::string company_name() const { return fetch("CompanyName"); }
		inline std::string file_description() const { return fetch("FileDescription"); }
		inline std::string file_version() const { return fetch("FileVersion"); }
		inline std::string internal_name() const { return fetch("InternalName"); }
		inline std::string legal_copyright() const { return fetch("LegalCopyright"); }
		inline std::string legal_trademarks() const { return fetch("LegalTrademarks"); }
		inline std::string original_filename() const { return fetch("OriginalFilename"); }
		inline std::string private_build() const { return fetch("PrivateBuild"); }
		inline std::string product_name() const { return fetch("ProductName"); }
		inline std::string product_version() const { return fetch("ProductVersion"); }
		inline std::string special_build() const { return fetch("SpecialBuild"); }

		inline bool any_flag() const { return has_flag(VS_FF_DEBUG | VS_FF_PATCHED | VS_FF_PRERELEASE | VS_FF_PRIVATEBUILD | VS_FF_SPECIALBUILD); }

		inline bool is_debug() const { return has_flag(VS_FF_DEBUG); }
		inline bool is_patched() const { return has_flag(VS_FF_PATCHED); }
		inline bool is_prerelease() const { return has_flag(VS_FF_PRERELEASE); }
		inline bool is_private_build() const { return has_flag(VS_FF_PRIVATEBUILD); }
		inline bool is_special_build() const { return has_flag(VS_FF_SPECIALBUILD); }

		std::string flag_str() const;
	private:
		bool _ready;
		std::filesystem::path _filename;
		BYTE* _buffer;
		DWORD _buf_size;
		Language* _lang_list;
		DWORD _lang_count;
		DWORD _lang_sel;
		VS_FIXEDFILEINFO* _info;

		bool has_flag(DWORD AFlag) const { return (_info->dwFileFlags & _info->dwFileFlagsMask) & AFlag; }
	};

}

std::string CyTools::CyVersion::Language::as_string() const {
	char buffer[10];

	sprintf(buffer, "%04x%04x", lang_id, charset);

	return buffer;
}

bool CyTools::CyVersion::load(const std::filesystem::path& ABinName) {
	std::string filename = ABinName.string();
	bool result = false;
	DWORD size;
	DWORD dummy;

	_filename = ABinName;
	_lang_list = nullptr;
	_lang_count = 0;
	_lang_sel = 0;
	_info = nullptr;

	size = GetFileVersionInfoSizeA(filename.c_str(), &dummy);
	if (size != 0) {
		if (size > _buf_size) {
			void* new_buffer = std::realloc(_buffer, size);

			if (new_buffer != nullptr) {
				_buffer = (BYTE*) new_buffer;
				_buf_size = size;
			}
		}

		if ((_buf_size >= size) && GetFileVersionInfoA(filename.c_str(), dummy, _buf_size, _buffer)) {
			void* pointer;
			UINT size;

			if (VerQueryValueA(_buffer, "\\", &pointer, &size)) {
				_info = (VS_FIXEDFILEINFO*) pointer;

				if (VerQueryValueA(_buffer, "\\VarFileInfo\\Translation", &pointer, &size)) {
					_lang_list = (Language*) pointer;
					_lang_count = size / sizeof(Language);

					result = true;
				}
			}
		}
	}

	_ready = result;
	return result;
}

std::string CyTools::CyVersion::fetch(const std::string& AName) const {
	std::string result = "";
	std::string query;
	void* pointer;
	UINT size;

	query = "\\StringFileInfo\\" + _lang_list[_lang_sel].as_string() + "\\" + AName;
	if (VerQueryValueA(_buffer, query.c_str(), &pointer, &size))
		result = std::string((char*) pointer, size - 1);

	return result;
}

std::string CyTools::CyVersion::flag_str() const {
	DWORD flags = (_info->dwFileFlags & _info->dwFileFlagsMask);
	std::string result = "";
	bool punch = false;

	if (flags & VS_FF_PATCHED) {
		result = "Patched";
		punch = true;
	}

	if (flags & VS_FF_DEBUG) {
		if (punch)
			result += ", ";

		result += "Debug";
		punch = true;
	}

	if (flags & VS_FF_PRERELEASE) {
		if (punch)
			result += ", ";

		result += "Pre-release";
		punch = true;
	}

	if (flags & VS_FF_PRIVATEBUILD) {
		if (punch)
			result += ", ";

		result += "Private build";
		punch = true;
	}

	if (flags & VS_FF_SPECIALBUILD) {
		if (punch)
			result += ", ";

		result += "Special build";
	}

	return result;
}

#endif


